#!/bin/bash

# ================= 設定區 =================
# 無需手動設定列表，自動掃描當前目錄下的 *.sql 檔
# ==========================================

echo "========================================"
echo "      MySQL 自動掃描匯入工具 (Auto)      "
echo "========================================"

# 0. 掃描 SQL 檔案
# 使用 glob 模式找出所有 .sql 檔案
SQL_FILES=(*.sql)

# 檢查是否有找到檔案
if [ ! -e "${SQL_FILES[0]}" ]; then
    echo "❌ 錯誤：當前目錄下找不到任何 .sql 檔案！"
    exit 1
fi

echo "發現以下 SQL 檔案，將自動建立對應資料庫："
for FILE in "${SQL_FILES[@]}"; do
    # 去除副檔名 .sql 取得 DB 名稱
    DB_NAME="${FILE%.sql}"
    echo " - $FILE  ->  DB: $DB_NAME"
done
echo "========================================"


# 1. 詢問連線資訊
read -p "請輸入 MySQL Host (預設 127.0.0.1): " DB_HOST
DB_HOST=${DB_HOST:-127.0.0.1}

read -p "請輸入 MySQL Port (預設 3306): " DB_PORT
DB_PORT=${DB_PORT:-3306}

read -p "請輸入使用者名稱 (預設 root): " DB_USER
DB_USER=${DB_USER:-root}

read -s -p "請輸入密碼: " DB_PASS
echo "" 

echo "----------------------------------------"
echo "正在測試連線..."

export MYSQL_PWD=$DB_PASS
if ! mysqladmin -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" ping --silent; then
    echo "❌ 連線失敗！請檢查 Host, Port, User 或 Password。"
    exit 1
fi
echo "✅ 連線成功！"
echo "----------------------------------------"

# 2. 開始自動處理
for SQL_FILE in "${SQL_FILES[@]}"; do
    # 從檔名解析資料庫名稱 (例如 smartfds_trans.sql -> smartfds_trans)
    DB_NAME="${SQL_FILE%.sql}"

    echo "正在處理: [ $DB_NAME ] (來源: $SQL_FILE)"

    # A. 建立資料庫 (utf8)
    echo "   1. 建立資料庫..."
    # 注意：這裡加上了 `` 反引號包住 DB_NAME，避免檔名有特殊字元報錯
    mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" -e "CREATE DATABASE IF NOT EXISTS \`$DB_NAME\` CHARACTER SET utf8 COLLATE utf8_general_ci;"

    if [ $? -ne 0 ]; then
        echo "   ❌ 建立資料庫失敗，跳過。"
        continue
    fi

    # B. 匯入
    echo "   2. 匯入資料..."
    mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" "$DB_NAME" < "$SQL_FILE"

    if [ $? -eq 0 ]; then
        echo "   ✅ 匯入成功！"
    else
        echo "   ❌ 匯入失敗！"
    fi
    echo "----------------------------------------"
done

unset MYSQL_PWD
echo "所有作業完成。"


