#!/bin/bash

# ================= 設定區 =================
# 要匯出的資料庫列表 (請在此維護要備份的 DB 名稱)
DATABASES=(
    "dev_smartfds_sso"
    "dev_smartfds_adm"
    "dev_smartfds_trans"
    "dev_acqfd_adm"
)
# ==========================================

echo "========================================"
echo "      MySQL DDL 結構匯出工具            "
echo "========================================"

# 1. 詢問連線資訊
read -p "請輸入 MySQL Host (預設 192.168.10.209): " DB_HOST
DB_HOST=${DB_HOST:-192.168.10.209} # 如果直接按 Enter，使用預設值

read -p "請輸入使用者名稱 (預設 veriid): " DB_USER
DB_USER=${DB_USER:-veriid}

# 提示輸入密碼 (隱藏輸入內容)
read -s -p "請輸入密碼: " DB_PASS
echo "" # 換行

echo "----------------------------------------"
echo "正在測試連線..."

# 檢查 mysqldump 是否存在
if ! command -v mysqldump &> /dev/null; then
    echo "錯誤: 系統中找不到 mysqldump 指令，請先安裝 MySQL Client。"
    exit 1
fi

# 簡單測試連線 (利用 mysqladmin)
export MYSQL_PWD=$DB_PASS
if ! mysqladmin -h "$DB_HOST" -u "$DB_USER" ping --silent; then
    echo "❌ 連線失敗！請檢查 Host, User 或 Password。"
    exit 1
fi
echo "✅ 連線成功，開始匯出作業..."
echo "----------------------------------------"

# 2. 迴圈處理每個資料庫
for DB_NAME in "${DATABASES[@]}"; do
    # 產生檔名：去除開頭的 "dev_" 並加上 .sql
    FILE_NAME="${DB_NAME#dev_}.sql"

    echo "正在處理: [ ${DB_NAME} ] -> ${FILE_NAME} ..."

    # 執行 mysqldump 並過濾內容
    # 1. --no-data: 只匯出結構
    # 2. sed 1: 去除 CHARSET/COLLATE
    # 3. sed 2: 去除 AUTO_INCREMENT=數字
    
    mysqldump -h "$DB_HOST" -u "$DB_USER" --no-data "$DB_NAME" \
    | sed -E 's/(DEFAULT )?(CHARSET|COLLATE)[ =][a-zA-Z0-9_]+//g' \
    | sed -E 's/AUTO_INCREMENT=[0-9]+//g' \
    > "$FILE_NAME"

    if [ $? -eq 0 ]; then
        echo "   ✅ 匯出成功"
    else
        echo "   ❌ 匯出失敗 (請檢查資料庫名稱是否正確)"
    fi
    echo "----------------------------------------"
done

# 清除密碼變數
unset MYSQL_PWD

echo "所有作業完成！檔案已儲存於當前目錄。"

